import React from 'react';
import { render, screen } from '@testing-library/react';
import { userEvent } from '@testing-library/user-event';
import { QuantitySelector } from '../QuantitySelector';

describe('QuantitySelector', () => {
  const mockOnIncrement = jest.fn();
  const mockOnDecrement = jest.fn();

  afterEach(() => {
    jest.clearAllMocks();
  });

  it('renders the current value', () => {
    render(
      <QuantitySelector
        value={5}
        onIncrement={mockOnIncrement}
        onDecrement={mockOnDecrement}
      />
    );
    expect(screen.getByTitle('Current quantity')).toHaveTextContent('5');
  });

  it('calls onIncrement when increment button is clicked', async () => {
    const user = userEvent.setup();
    render(
      <QuantitySelector
        value={5}
        onIncrement={mockOnIncrement}
        onDecrement={mockOnDecrement}
      />
    );
    
    await user.click(screen.getByLabelText('Increase quantity'));
    expect(mockOnIncrement).toHaveBeenCalledTimes(1);
  });

  it('calls onDecrement when decrement button is clicked', async () => {
    const user = userEvent.setup();
    render(
      <QuantitySelector
        value={5}
        onIncrement={mockOnIncrement}
        onDecrement={mockOnDecrement}
      />
    );
    
    await user.click(screen.getByLabelText('Decrease quantity'));
    expect(mockOnDecrement).toHaveBeenCalledTimes(1);
  });

  it('disables decrement button when at minimum value', () => {
    render(
      <QuantitySelector
        value={1}
        onIncrement={mockOnIncrement}
        onDecrement={mockOnDecrement}
        min={1}
      />
    );
    
    expect(screen.getByLabelText('Decrease quantity')).toBeDisabled();
  });

  it('disables increment button when at maximum value', () => {
    render(
      <QuantitySelector
        value={9}
        onIncrement={mockOnIncrement}
        onDecrement={mockOnDecrement}
        max={9}
      />
    );
    
    expect(screen.getByLabelText('Increase quantity')).toBeDisabled();
  });

  it('respects custom min value', () => {
    render(
      <QuantitySelector
        value={5}
        onIncrement={mockOnIncrement}
        onDecrement={mockOnDecrement}
        min={5}
      />
    );
    
    expect(screen.getByLabelText('Decrease quantity')).toBeDisabled();
  });

  it('respects custom max value', () => {
    render(
      <QuantitySelector
        value={15}
        onIncrement={mockOnIncrement}
        onDecrement={mockOnDecrement}
        max={15}
      />
    );
    
    expect(screen.getByLabelText('Increase quantity')).toBeDisabled();
  });

  it('enables both buttons when value is between min and max', () => {
    render(
      <QuantitySelector
        value={5}
        onIncrement={mockOnIncrement}
        onDecrement={mockOnDecrement}
        min={1}
        max={9}
      />
    );
    
    expect(screen.getByLabelText('Decrease quantity')).not.toBeDisabled();
    expect(screen.getByLabelText('Increase quantity')).not.toBeDisabled();
  });
});
