/* eslint-disable react/jsx-no-comment-textnodes */
'use client';

import { useState } from 'react';
import Image from 'next/image';
import { useProductQuery } from '@repo/graphql-types';
import {
  useCart,
  QuantitySelector,
  ProductCard,
  ProductSpecs,
  Button,
  LoadingSpinner,
  formatPrice,
  CartPopover,
  getProductImageUrl,
} from '@repo/kraken-ui';
import styles from './product.module.css';

export default function Product() {
  const [quantity, setQuantity] = useState(1);
  const [isCartOpen, setIsCartOpen] = useState(false);
  const { itemCount, addToCart } = useCart();
  const { data, loading, error } = useProductQuery({
    variables: { id: '1' },
  });

  if (loading) return <LoadingSpinner />;
  if (error) return <div className={styles.error}>Error loading product: {error.message}</div>;
  if (!data?.Product) return <div className={styles.error}>Product not found</div>;

  const product = data.Product;

  const handleIncrement = () => {
    setQuantity((prev) => prev + 1);
  };

  const handleDecrement = () => {
    if (quantity > 1) {
      setQuantity((prev) => prev - 1);
    }
  };

  const handleAddToCart = () => {
    addToCart(
      {
        id: product.id,
        name: product.name || '',
        price: product.price || 0,
        image: product.img_url || '',
      },
      quantity
    );
    setQuantity(1);
  };

  const specifications = [
    { label: 'Brand', value: product.brand || '' },
    { label: 'Item weight (g)', value: product.weight?.toString() || '' },
    {
      label: 'Dimensions (cm)',
      value: `${product.height} x ${product.width} x ${product.length}`,
    },
    { label: 'Item Model number', value: product.model_code || '' },
    { label: 'Colour', value: product.colour || '' },
  ];

  return (
    <div className={styles.container}>
      <header className={styles.header}>
        <div className={styles.headerContent}>
          <Image
            src="/octopus-logo.svg"
            alt="Octopus Energy"
            className={styles.logo}
            width={470}
            height={67}
            priority
          />
          <button className={styles.basket} onClick={() => setIsCartOpen(!isCartOpen)}>
            <Image src="/basket.svg" alt="Basket" className={styles.basketIcon} width={32} height={32} />
            {itemCount > 0 && (
              <span className={styles.basketCount} title="Basket items">
                {itemCount}
              </span>
            )}
          </button>
          <CartPopover isOpen={isCartOpen} onClose={() => setIsCartOpen(false)} />
        </div>
      </header>

      <main className={styles.main}>
        <div className={styles.contentWrapper}>
          <ProductCard>
            <div className={styles.productImage}>
              <Image 
                src={getProductImageUrl(product.img_url || '')}
                alt={product.name || ''} 
                width={740}
                height={740}
                style={{ objectFit: 'contain', width: '100%', height: 'auto' }}
              />
            </div>
          </ProductCard>

          <div className={styles.productInfo}>
            <h1 className={styles.title}>{product.name}</h1>
            <p className={styles.subtitle}>
              {product.power} // Packet of {product.quantity}
            </p>

            <div className={styles.priceSection}>
              <p className={styles.price}>{formatPrice(product.price || 0)}</p>
              <div className={styles.quantitySection}>
                <label className={styles.qtyLabel}>Qty</label>
                <QuantitySelector
                  value={quantity}
                  onIncrement={handleIncrement}
                  onDecrement={handleDecrement}
                  min={1}
                />
              </div>
            </div>

            <Button onClick={handleAddToCart}>Add to cart</Button>
          </div>
        </div>
      </main>

      <section className={styles.descriptionSection}>
        <div className={styles.contentWrapper}>
          <h2>Description</h2>
          <p>{product.description}</p>
        </div>
      </section>

      <section className={styles.specsSection}>
        <div className={styles.contentWrapper}>
          <ProductSpecs specifications={specifications} />
        </div>
      </section>

      <footer className={styles.footer}>
        <div className={styles.contentWrapper}>
          <p>
            Octopus Energy Ltd is a company registered in England and Wales.
            Registered number: 09263424. Registered office: 33 Holborn, London,
            EC1N 2HT. Trading office: 20-24 Broadwick Street, London, W1F 8HT
          </p>
        </div>
      </footer>
    </div>
  );
}
